import LoadingCircle from "@/components/icons/loading.circle";
import { Button } from "@/components/ui/button";
import { DialogClose, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormDescription, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { userGroupDescriptions } from "@/config/constants";
import { errorToStringFormatter } from "@/lib/formatters/error-formatters";
import { TransformedAgencies } from "@/types/transformers";
import { ComponentWithClassName } from "@/types/utils";
import { AppUserGroup, userGroupSchema } from "@/zod/amplify.schema";
import { zodResolver } from "@hookform/resolvers/zod";
import { Asterisk } from "lucide-react";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { inviteUser } from "../lib/actions";
import { SelectMulti } from "@/components/ui/select.multi";
import { revalidateListAllUsers } from "../hooks/use-list-all-users";

const INTERNAL__AGENCY_CATCH_ALL_ID = 'MDOT_AGENCY_CATCH_ALL';
const INTERNAL__delimiter = ',';

export function WorkflowDialogInvite({ setClose, agencies, className }: ComponentWithClassName<{ setClose: () => void, agencies: TransformedAgencies }>) {
  const agencyOptions = agencies.map((a) => {
    return {
      label: `${a.name}`,
      value: a.gtfsId,
    }
  })

  agencyOptions.unshift({
    label: 'Michigan Department of Transportation',
    value: INTERNAL__AGENCY_CATCH_ALL_ID,
  })

  const agencyIds = agencyOptions.map((a) => a.value)
  agencyIds.push(INTERNAL__AGENCY_CATCH_ALL_ID)
  const defaultOptions = [agencyIds[0]];
  const [errors, setErrors] = useState<(Error | string)[] | undefined>(undefined)
  const formSchema = z.object({
    user_associated_agency: z.array(z.string()).min(1).max(24),
    user_group: userGroupSchema,
    user_email: z.string().min(2).max(64),
  });

  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      user_associated_agency: [agencyOptions[0]?.value],
      user_email: "",
      user_group: "VIEWER",
    }
  })
  const { formState: { isSubmitting } } = form

  function clearAgencies() {
    form.setValue('user_associated_agency', defaultOptions)
  }

  if (!agencyOptions || agencyOptions.length === 0) {
    return (
      <DialogContent className={className}>
        <DialogHeader>
          <DialogTitle>Unable to accept members...</DialogTitle>
          <DialogDescription>
            We are currently not accepting new members at this time. Try again at a later date.
          </DialogDescription>
        </DialogHeader>

        <div className="flex flex-row gap-2 justify-end">
          <DialogClose asChild>
            <Button variant={"secondary"}>Close</Button>
          </DialogClose>
        </div>
      </DialogContent>
    )
  }

  return (
    <DialogContent className={className}>
      <DialogHeader>
        <DialogTitle>Invite a new member.</DialogTitle>
        <DialogDescription>
          Members are given permissions to view, edit, or delete feeds and other services associated with the Trip Planner.
        </DialogDescription>
      </DialogHeader>

      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(async (values) => {
            const res = await inviteUser(values.user_email, values.user_associated_agency.join(INTERNAL__delimiter), values.user_group)
            await revalidateListAllUsers()

            if (!!res.errors && res.errors.length > 0) {
              setErrors(res.errors.map((e) => e.message))
              return;
            }

            setClose()
          })}
          className="space-y-4"
        >
          <FormField
            control={form.control}
            name="user_associated_agency"
            render={({ field: { value, onChange, ...field } }) => (
              <FormItem>
                <FormLabel className="flex flex-row gap-0.5">Agency identifier <Asterisk className="text-destructive size-3" /></FormLabel>
                <FormControl>
                  <SelectMulti
                    {...field}
                    selectedOptions={value}
                    defaultOptions={defaultOptions}
                    onSelectionChanged={(value) => onChange(value)}
                    options={agencyOptions}
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="user_group"
            render={({ field: { value, onChange, ...field } }) => (
              <FormItem>
                <FormLabel className="flex flex-row gap-0.5">User role <Asterisk className="text-destructive size-3" /></FormLabel>
                <FormControl>
                  <Select
                    onValueChange={(value) => {
                      if (value === "ADMIN" satisfies AppUserGroup) {
                        form.setValue("user_associated_agency", agencyIds)
                      } else if (value === "VIEWER" satisfies AppUserGroup) [
                        clearAgencies()
                      ]
                      onChange(value)
                    }}
                    defaultValue={value}
                  >
                    <SelectTrigger>
                      <SelectValue
                        placeholder="Select associated agency"
                        className="text-start w-full"
                        {...field}
                      />
                    </SelectTrigger>
                    <SelectContent>
                      {userGroupDescriptions.map((version) => (
                        <SelectItem
                          key={version.display}
                          value={version.value}
                          className="flex flex-col gap-1 justify-start items-start"
                        >
                          {version.display}{" "}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />


          <FormField
            control={form.control}
            name="user_email"
            render={({ field }) => (
              <FormItem>
                <FormLabel className="flex flex-row gap-0.5">Email <Asterisk className="text-destructive size-3" /></FormLabel>
                <FormControl>
                  <Input
                    placeholder="email@acme.com"
                    type="email"
                    {...field} />
                </FormControl>
                <FormDescription>This email will recieve an email for verification.</FormDescription>
                <FormMessage />
              </FormItem>
            )}
          />

          {
            !!errors && (
              errors.map((e, i) => {
                return (
                  <div key={i} className="text-destructive text-xs" > {errorToStringFormatter(e)} </div>
                )
              })
            )
          }

          <div className="flex flex-row gap-2 justify-end">
            <DialogClose asChild>
              <Button variant={"secondary"}>Cancel</Button>
            </DialogClose>

            <Button type="submit" disabled={isSubmitting}>
              Invite member{" "}
              {isSubmitting && <LoadingCircle className="ml-2" />}
            </Button>
          </div>
        </form>
      </Form>
    </DialogContent>
  )
}

